<?php
/**
 * Plugin Name: Best Posts Affiliate Disclosure
 * Plugin URI: https://digitalhubit.com/
 * Description: Shows affiliate disclosure only on blog posts with titles starting with "Best"
 * Version: 1.0.0
 * Author: Tariqul Islam
 * License: GPL v2 or later
 * Text Domain: best-posts-disclosure
 */

// Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

class BestPostsAffiliateDisclosure {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // Hook into post content - at the very beginning
        add_filter('the_content', array($this, 'add_affiliate_disclosure'), 1);
        
        // Alternative hook for better positioning after title/meta
        add_action('generate_after_entry_header', array($this, 'add_disclosure_after_header'));
        
        // Add admin menu for settings
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
    }
    
    /**
     * Add disclosure after entry header (GeneratePress specific)
     */
    public function add_disclosure_after_header() {
        // Only show on single posts
        if (!is_single()) {
            return;
        }
        
        // Get the post title
        $post_title = get_the_title();
        
        // Check if title starts with "Best" (case-insensitive)
        if (!$this->title_starts_with_best($post_title)) {
            return;
        }
        
        // Output the disclosure
        echo $this->get_disclosure_html();
    }
    
    /**
     * Add affiliate disclosure to posts starting with "Best"
     */
    public function add_affiliate_disclosure($content) {
        // Only show on single posts, not pages or archives
        if (!is_single()) {
            return $content;
        }
        
        // Get the post title
        $post_title = get_the_title();
        
        // Check if title starts with "Best" (case-insensitive)
        if (!$this->title_starts_with_best($post_title)) {
            return $content;
        }
        
        // Get disclosure position setting
        $position = get_option('bpad_position', 'after_header');
        
        // If position is set to use GeneratePress hook, don't add to content
        if ($position === 'after_header') {
            return $content;
        }
        
        $disclosure_html = $this->get_disclosure_html();
        
        // Add disclosure based on position setting
        switch ($position) {
            case 'before_content':
                return $disclosure_html . $content;
            case 'after_content':
                return $content . $disclosure_html;
            default:
                return $content;
        }
    }
    
    /**
     * Check if title starts with "Best"
     */
    private function title_starts_with_best($title) {
        $title = trim($title);
        return (stripos($title, 'best') === 0);
    }
    
    /**
     * Generate the disclosure HTML
     */
    private function get_disclosure_html() {
        $custom_text = get_option('bpad_custom_text', 'We earn from qualifying purchases through some links here, but we only recommend what we truly love. No fluff, just honest picks!');
        
        ob_start();
        ?>
        <div class="affiliate-disclosure bpad-disclosure">
            <div class="disclosure-icon">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z" fill="currentColor"/>
                </svg>
            </div>
            <div class="disclosure-content">
                <p class="disclosure-text">
                    <strong>Affiliate Disclosure:</strong> <?php echo esc_html($custom_text); ?>
                </p>
            </div>
        </div>
        
        <style>
        .bpad-disclosure {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 1px solid #dee2e6;
            border-left: 4px solid #007cba;
            border-radius: 8px;
            padding: 16px 20px;
            margin: 20px 0;
            display: flex;
            align-items: flex-start;
            gap: 12px;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .bpad-disclosure:hover {
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            transform: translateY(-1px);
        }
        
        .bpad-disclosure .disclosure-icon {
            color: #007cba;
            flex-shrink: 0;
            margin-top: 2px;
            opacity: 0.8;
        }
        
        .bpad-disclosure .disclosure-content {
            flex: 1;
        }
        
        .bpad-disclosure .disclosure-text {
            margin: 0;
            color: #495057;
            font-size: 14px;
            line-height: 1.5;
            font-weight: 400;
        }
        
        .bpad-disclosure .disclosure-text strong {
            color: #343a40;
            font-weight: 600;
        }
        
        @media (max-width: 768px) {
            .bpad-disclosure {
                padding: 14px 16px;
                margin: 16px 0;
                gap: 10px;
            }
            
            .bpad-disclosure .disclosure-text {
                font-size: 13px;
            }
        }
        
        @media (prefers-color-scheme: dark) {
            .bpad-disclosure {
                background: linear-gradient(135deg, #2d3748 0%, #1a202c 100%);
                border-color: #4a5568;
                border-left-color: #63b3ed;
            }
            
            .bpad-disclosure .disclosure-icon {
                color: #63b3ed;
            }
            
            .bpad-disclosure .disclosure-text {
                color: #e2e8f0;
            }
            
            .bpad-disclosure .disclosure-text strong {
                color: #f7fafc;
            }
        }
        </style>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            'Best Posts Affiliate Disclosure',
            'Affiliate Disclosure',
            'manage_options',
            'best-posts-disclosure',
            array($this, 'options_page')
        );
    }
    
    /**
     * Settings initialization
     */
    public function settings_init() {
        register_setting('bpad_settings', 'bpad_custom_text');
        register_setting('bpad_settings', 'bpad_position');
        register_setting('bpad_settings', 'bpad_keywords');
        
        add_settings_section(
            'bpad_settings_section',
            'Disclosure Settings',
            array($this, 'settings_section_callback'),
            'bpad_settings'
        );
        
        add_settings_field(
            'bpad_custom_text',
            'Disclosure Text',
            array($this, 'custom_text_render'),
            'bpad_settings',
            'bpad_settings_section'
        );
        
        add_settings_field(
            'bpad_position',
            'Display Position',
            array($this, 'position_render'),
            'bpad_settings',
            'bpad_settings_section'
        );
        
        add_settings_field(
            'bpad_keywords',
            'Additional Keywords',
            array($this, 'keywords_render'),
            'bpad_settings',
            'bpad_settings_section'
        );
    }
    
    public function custom_text_render() {
        $value = get_option('bpad_custom_text', 'We earn from qualifying purchases through some links here, but we only recommend what we truly love. No fluff, just honest picks!');
        echo '<textarea name="bpad_custom_text" rows="3" cols="50" class="large-text">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">Enter your custom affiliate disclosure text.</p>';
    }
    
    public function position_render() {
        $value = get_option('bpad_position', 'after_header');
        echo '<select name="bpad_position">';
        echo '<option value="after_header"' . selected($value, 'after_header', false) . '>After Title/Meta (Recommended)</option>';
        echo '<option value="before_content"' . selected($value, 'before_content', false) . '>Before Content</option>';
        echo '<option value="after_content"' . selected($value, 'after_content', false) . '>After Content</option>';
        echo '</select>';
        echo '<p class="description">Choose where to display the disclosure. "After Title/Meta" shows it right after the post title and date, before content starts.</p>';
    }
    
    public function keywords_render() {
        $value = get_option('bpad_keywords', '');
        echo '<input type="text" name="bpad_keywords" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">Additional keywords to trigger disclosure (comma-separated). Example: top, review, guide</p>';
    }
    
    public function settings_section_callback() {
        echo '<p>Configure your affiliate disclosure settings for posts starting with "Best".</p>';
    }
    
    /**
     * Options page HTML
     */
    public function options_page() {
        ?>
        <div class="wrap">
            <h1>Best Posts Affiliate Disclosure Settings</h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('bpad_settings');
                do_settings_sections('bpad_settings');
                submit_button();
                ?>
            </form>
            
            <div class="card" style="margin-top: 20px;">
                <h2>Preview</h2>
                <p>This is how your disclosure will look:</p>
                <?php echo $this->get_disclosure_html(); ?>
            </div>
            
            <div class="card" style="margin-top: 20px;">
                <h2>How it works</h2>
                <ul>
                    <li>✅ Shows disclosure only on single blog posts</li>
                    <li>✅ Only displays if post title starts with "Best" (case-insensitive)</li>
                    <li>✅ Fully customizable text and position</li>
                    <li>✅ Mobile responsive design</li>
                    <li>✅ Dark mode support</li>
                </ul>
            </div>
        </div>
        <?php
    }
}

// Initialize the plugin
new BestPostsAffiliateDisclosure();

// Activation hook
register_activation_hook(__FILE__, 'bpad_activate');
function bpad_activate() {
    // Set default options
    add_option('bpad_custom_text', 'We earn from qualifying purchases through some links here, but we only recommend what we truly love. No fluff, just honest picks!');
    add_option('bpad_position', 'after_header');
    add_option('bpad_keywords', '');
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'bpad_deactivate');
function bpad_deactivate() {
    // Clean up if needed
}
?>